<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Class containing data for myprofile block.
 *
 * @package    block_myprofile
 * @copyright  2018 Mihail Geshoski <mihail@moodle.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace block_mygroupcourse\output;

defined('MOODLE_INTERNAL') || die();

use renderable;
use renderer_base;
use templatable;
use moodle_url;
/**
 * Class containing data for myprofile block.
 *
 * @copyright  2018 Mihail Geshoski <mihail@moodle.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class main implements renderable, templatable {

    /**
     * @var object An object containing the configuration information for the current instance of this block.
     */
    protected $config;

    /**
     * @var string An containing id current categorie for select courses.
     */
    private $actual_categorie;

    /**
     * Constructor.
     *
     * @param object $config An object containing the configuration information for the current instance of this block.
     */
    public function __construct() {
    }

    /**
     * Export this data so it can be used as the context for a mustache template.
     *
     * @param \renderer_base $output
     * @return stdClass
     */
    public function export_for_template(renderer_base $output) {
        global $USER, $OUTPUT;
        $nocoursesurl = $OUTPUT->image_url('courses', 'block_recentlyaccessedcourses')->out(false);
        $categories=$this->categories_course();
        $courses =[];
        if(count($categories)>0){

            $courses =$this->data_courses();
        }
        //$data = new \stdClass();
        
        $data=[
            'have_categories'=> (count($categories)>0 ? true:false),
            'categories'=>$categories,
            'courses' =>$courses,
            'have_courses'=>(count($courses)>0 ? true:false),
            'nocoursesurl'=> $nocoursesurl,
            'have_nocurse_img'=>true
        ];
     
        return $data;
    }



    public function categories_course(){
        global $DB, $USER;
        $result = [];
        $sql = "
        SELECT ct.id,ct.name as categoria
        FROM {user} u
        JOIN {role_assignments} a ON (a.userid=u.id)
        JOIN {role} r ON (r.id = a.roleid)
        JOIN {context} t ON (t.id = a.contextid)
        JOIN {course} c ON (c.id =t.instanceid AND c.visible = 1)
        JOIN {course_categories} ct ON (c.category=ct.id)
        WHERE t.contextlevel= 50
        AND u.id=".$USER->id."
        GROUP BY ct.id,ct.name
        ORDER BY ct.sortorder";
        
        $categories = $DB->get_records_sql($sql);
        $first = true;
        foreach($categories as $categorie){
            array_push($result,['categorie'=>array_merge((array)$categorie,['active'=> $first])]);
            if($first){
                $this->actual_categorie= $categorie->id;
                $first=false;
            }
        }

        return $result;
    }

    public function data_courses($id_actual_course=null){
        global $DB, $USER, $OUTPUT;
        if(!is_null($id_actual_course)){
            $this->actual_categorie=$id_actual_course;
        }

        $result=[];
        $sql = "SELECT c.id,c.fullname as curso,c.shortname as codigo,ct.name as categoria
        FROM {user} u
        JOIN {role_assignments} a ON (a.userid=u.id)
        JOIN {role} r ON (r.id = a.roleid)
        JOIN {context} t ON (t.id = a.contextid)
        JOIN {course} c ON (c.id =t.instanceid AND c.visible = 1)
        JOIN {course_categories} ct ON (c.category=ct.id)
        WHERE t.contextlevel= 50
        AND u.id=".$USER->id."
        AND ct.id=".$this->actual_categorie."
        ORDER BY ct.sortorder, c.sortorder";
        $data_courses = $DB->get_records_sql($sql);
        $categoryName="";
        foreach($data_courses as $course){    
            $c = get_course($course->id);     
          
            array_push($result,['course'=>$this->get_other_values($c)]);
            $categoryName=$course->categoria;
        }   
        array_push($result,['categoryName'=>$categoryName] );
        return $result;   
    }



    protected function get_other_values($course) {
        global $CFG,$OUTPUT;
        $courseimage = self::get_course_image($course);
        if (!$courseimage) {
            $courseimage = $OUTPUT->get_generated_image_for_id($course->id);
        }

        $progress = self::get_course_progress($course);
        $hasprogress = false;
        if ($progress === 0 || $progress > 0) {
            $hasprogress = true;
        }
        $progress = floor($progress);
        $coursecategory = \core_course_category::get($course->category, MUST_EXIST, true);
        $coursename = format_string($course->fullname, true,
        array('context' => \context_course::instance($course->id)));
     
   
        return array(
            'fullnamedisplay' => get_course_display_name_for_list($course),
            'viewurl' => (new moodle_url('/course/view.php', array('id' => $course->id)))->out(false),
            'courseimage' => $courseimage,
            'progress' => $progress,
            'hasprogress' => $hasprogress,
            'isfavourite' => $this->related['isfavourite'],
            'hidden' => boolval(get_user_preferences('block_myoverview_hidden_course_' . $course->id, 0)),
            'showshortname' => $CFG->courselistshortnames ? true : false,
            'coursecategory' => $coursecategory->name,
            'id'=>$course->id,
            'coursename' =>$coursename,
            'fullname'=>$course->fullname,
            'shortname'=>$shortname
        );
    }
    
    /**
     * Get the course image if added to course.
     *
     * @param object $course
     * @return string url of course image
     */
    public static function get_course_image($course) {
        global $CFG;
        $courseinlist = new \core_course_list_element($course);
        foreach ($courseinlist->get_course_overviewfiles() as $file) {
            if ($file->is_valid_image()) {
                $pathcomponents = [
                    '/pluginfile.php',
                    $file->get_contextid(),
                    $file->get_component(),
                    $file->get_filearea() . $file->get_filepath() . $file->get_filename()
                ];
                $path = implode('/', $pathcomponents);
                return (new moodle_url($path))->out();
            }
        }
        return false;
  
    }

    /**
     * Get the course progress percentage.
     *
     * @param object $course
     * @return int progress
     */
    public static function get_course_progress($course) {
        return \core_completion\progress::get_course_progress_percentage($course);
    }
    

}
